/* matrix.c j.p.zilla -3 - a few general matrix routines
 * modified
 * jan01	ansi update
 * 9apr/2       ansi update
 *
 * package: Mat
 * todo:
 * include solve(), renamed to MatSolve() for conflicts.
 *
 * typing philosophy:
 * matrices and vectors allocated with Alloc() have type fields
 * before and after the data, so the data can be checked for
 * overwriting. routines other than Free() do not check though,
 * so arrays from other sources (not Allocated) can be used in 
 * these routines.
 *
 */

/*@DOCINIT
 (@HEADERFILE matrix.H)
 (@MANPAGE matrix.3 Matrix 3 unknown matrix library)
*/

/* declare matrix as integer addresses if we Really need to use this */
/*@LINKCTYPE (matrix I) */

/*@SECTION  DESCRIPTION
These routines are compiled in single and double precision versions.
The calling program should #define FLTPREC or DBLPREC as desired 
before #including matrix.h to get the appropriate routines.
Determinants are always double precision.
*/


#ifdef MATRIXINCLUDE /*%%%%%%%%%%%% (contents of matrix.h) %%%%%%%%%%%%*/

#ifndef MATRIX
# define MATRIX

# if (!(FLTPREC|DBLPREC))
  macroerror: #define FLTPREC or DBLPREC
# endif
# ifdef Flotype
   macroerror_Flotype
# endif

#ifdef FLTPREC
# define Flotype float
#else  /*DBLPREC*/
# define Flotype double
# define MatAlloc	dMatAlloc
# define MatFree	dMatFree
# define MatPrint	dMatPrint
# define MatCopy	dMatCopy
# define MatMul 	dMatMul
# define MatXpose	dMatXpose
# define MatInvert	dMatInvert
# define MatInvert2	dMatInvert2
# define MatSolve	dMatSolve
# define MatRandom	dMatRandom
  /* Vector routines */
# define VecAlloc	dVecAlloc
# define VecFree	dVecFree
# define VecPrint	dVecPrint
# define VecCopy	dVecCopy
# define MatVecMul	dMatVecMul
# define VecRandom	dVecRandom
  /* MacKay */
# define mackaysolve	dmackaysolve
# define mackay		dmackay
#endif /*DBLPREC*/

/* Tracemat((msg,mat,dim)); */
#ifdef	ztrace
# define Tracemat(msg,mat,dim)  MatPrint(msg,mat,dim)
#else
# define Tracemat(msg,mat,dim) 
#endif	/*ztrace*/

extern Flotype *MatAlloc();
extern void
  MatFree(),MatPrint(),MatCopy(),MatMul(),MatXpose(),MatRandom();
extern Flotype MatInvert(),MatInvert2(),MatSolve();
/* Vector */
extern Flotype *VecAlloc();
extern void
  VecFree(),VecPrint(),VecCopy(),VecRandom(),
  MatVecMul();
/* MacKay*/
extern Flotype mackaysolve();
extern void mackay(),mackay_setapprox(),mackay_seterrthresh();

#endif /*MATRIX*/
#endif /*MATRIXINCLUDE%%%%%%%%%%%% (contents of matrix.h) %%%%%%%%%%%%*/

#include <theusual.h>
#include <math.h>
#include <matrix.h>
#include <rnd.h>

#define MatKey 'M'


/*@SECTION Entries*/

/*@DOCENTRY
(PROTO (Flotype *) MatAlloc (int4 m) (int4 n))
(USAGE Allocates storage for mxn matrix with a type-checking field.)
*/

Flotype *
MatAlloc(int4 m,int4 n)
{
    Flotype *M;

    M = (Flotype *)malloc((m*n+3) * sizeof(Flotype));

    /* set type-checking keys before,after the data
     * [key,length,...data...,key]
     */
    *((int4 *)(M+0)) = (int4)MatKey;
    *((int4 *)(M+1)) = m*n;	/* length */
    *((int4 *)(M+2+m*n)) = (int4)MatKey;

    return(M+2);
} /*Alloc*/


/*OLDENTRY
MatFree(M)
Frees M; complains if M was not allocated with MatAlloc.
*/

void MatFree(Flotype *M)
{
    int4 n;

    M -= 2;
    if (*((int4 *)(M+0)) != MatKey)  Zcodeerror("MatFree");

    n = *((int4 *)(M+1));
    Ztrace(("MatFree recovered length %d\n",n));

    if (*((int4 *)(M+2+n)) != (int4)MatKey)
	Zcodeerror("MatFree:corrupted matrix");

    free((char *)M);
} /*Free*/


/*OLDENTRY
MatPrint(char *msg,M,N)
Prints a square matrix of size N, prefaced by msg.
*/

void MatPrint(char *msg,Flotype A[],int4 n)
{
    int4 i,j,K;

    printf("%s\n",msg);

    K = 0;
    for( j=0; j < n; j++ ) {
	
	printf("[ ");

 	if (n <= 10)   
	    for(i=0;i<n;i++)
		printf("%.3f  ",A[K++]);

	else { /* elide */
	    for( i=0; i < 4; i++ ) 
		printf("%.3f  ",A[K+i]);
	    printf(" ... ");
	    for( i=n-2; i < n; i++ ) 
		printf("%.3f  ",A[K+i]);
	    K += n;
	}

    	printf(" ]\n");
    }

} /*Print*/



/*OLDENTRY
MatCopy(M1,M2,m,n)
Copies M1 to M2.
*/

void MatCopy(Flotype *M1,Flotype *M2,int4 m,int4 n)
{
    Zbcopy((char *)M1,(char *)M2,m*n*sizeof(Flotype));
} /*Copy*/


/*OLDENTRY
MatMul(M1,M2,M3,N)
Square matrix multiply M3 = M1*M2; M3 must be distinct from M1,M2.
*/

void MatMul(Flotype *A_,Flotype *B_,Flotype *C_,int4 N)
{
    register int4 r,c,i;
    register Flotype sum;

#   define A(i,j) A_[(i)*N+(j)]
#   define B(i,j) B_[(i)*N+(j)]
#   define C(i,j) C_[(i)*N+(j)]

    for( r=0; r < N; r++ ) {
	for( c=0; c < N; c++ ) {
	    sum = 0.0;
	    for( i=0; i < N; i++ ) {
		sum += A(r,i) * B(i,c);
	    }
	    C(r,c) = sum;
	}
    }

#   undef A
#   undef B
#   undef C

} /*matmul*/


/*OLDENTRY
MatXpose(M1,M2,N)
Transposes square matrix M1 into M2.
*/

void
MatXpose(Flotype *A_,Flotype *B_,int4 N)
{
    register int4 r,c;

#   define A(i,j) A_[(i)*N+(j)]
#   define B(i,j) B_[(i)*N+(j)]

    for( r=0; r < N; r++ ) {
	for( c=0; c < N; c++ ) {
	    B(r,c) = A(c,r);
	}
    }

#   undef A
#   undef B

} /*matxpose*/



/*OLDENTRY
double MatInvert(M,N)
Inverts M in place using Gauss-Jordan elimination, returning the determinant.
Code from 'invert' in IBM scientific subroutine library (public domain).
*/

double
MatInvert( Flotype mat[], int4 n )
{
#   define MAXMAT 100
    double det;
    register Flotype biga, hold;
    int4 l[MAXMAT], m[MAXMAT];
    register int4 i,j,k;
    int4 ij,jk,ik,ji, iz, ki, kj;
    int4 jp,jq,jr;
    int4 nk,kk;
 
    if( n >= MAXMAT ) {
      fprintf( stderr, "Matrix Invert : %d too large \n", n);
      return 0.0;
    }

    det = 1.0;
    nk = -n;
    for(k=0; k<n; k++ ) {
	nk=nk+n; kk=nk+k;
	/* 
	 *  Search for maximum value
	 */
	l[k] = k;
	m[k] = k;
	biga = mat[kk];
	for(j=k; j<n; j++ ) {
	    iz=n*j;
	    for(i=k; i<n; i++) {
		ij=iz+i;
		if( fabs(biga)-fabs(mat[ij]) < 0.0 ) {
		    biga=mat[ij];
		    l[k]=i;
		    m[k]=j;
		 }
             }
	 }

	/*
	 *  Interchange rows  
	 */
	j = l[k];
	if(j>k) {
	    ki=k-n;
	    for(i=0; i<n; i++ ) {
		ki=ki+n; ji=ki-k+j;
		hold= -mat[ki]; mat[ki]=mat[ji]; mat[ji]=hold;
	      }
	 }

	/*
	 *  Interchange columns 
	 */
	i=m[k];
	if(i>k) {
	    jp=n*i;
	    for(j=0; j<n; j++ ) {
		jk=nk+j; ji=jp+j;
		hold = -mat[jk]; mat[jk]=mat[ji]; mat[ji]=hold;
	     }
	 }

	/*
	 *	Divide column by minus pivot. The value of 
	 *	pivot is contained in biga.
	 */
	if(biga==0.0) {
	    fprintf(stderr,"Matrix Invert : singular matrix \n");
	    return 0.0;
	 }
	for(i=0; i<n; i++ ) {
	    if(i!=k) {
		ik = nk + i;
		mat[ik]=mat[ik]/(-biga);
	     }
	 }

	/*
	 *	Reduce matrix
	 */
	for(i=0; i<n; i++) {
	    ik=nk+i; hold = mat[ik]; ij=i-n;
	    for(j=0; j<n; j++ ) {
		ij=ij+n;
		if(i!=k) if(j!=k) {
		    kj=ij-i+k;
		    mat[ij]=hold*mat[kj]+mat[ij];
		 }
	     }
	 }

	/*
	 *  Divide row by pivot  
	 */
	kj=k-n;
	for(j=0; j<n; j++) {
	    kj=kj+n;
	    if(k!=j) mat[kj] = mat[kj]/biga;
	 }

	/*
	 *  Product of pivots  
	 */
	det *= biga;

	/*
	 *  Replace pivot by reciprocal 
	 */
	mat[kk]=1.0/biga;
    }

  /*  Final row and column interchange */
  for(k=n-2; k>=0; k--) {
      i=l[k];
      if(i>k) {
          jq=n*k; jr=n*i;
          for(j=0; j<n; j++ ) {
            jk=jq+j; ji=jr+j; 
            hold=mat[jk]; mat[jk] = -mat[ji]; mat[ji]=hold;
           }
       }

       j=m[k];
       if(j>k) {
	   ki=k-n;
           for(i=0; i<n; i++ ) {
	       ki=ki+n; ji=ki-k+j;
	       hold=mat[ki]; mat[ki]= -mat[ji]; mat[ji]=hold;
            }
        }
    }

  return( (Flotype)det );
} /*matinvert*/


/*OLDENTRY
double MatInvert2(M1,M2,N)
Inverts M1 into M2, returning the determinant.
*/

double
MatInvert2( Flotype *a, Flotype *b, int4 n )
{
  register int4 i, j;
  register Flotype *bb;
  bb = b;
  for( i=0; i<n; i++ ) for( j=0; j<n; j++ ) *bb++ = *a++;
  return( MatInvert( b, n ) );
} /*matinvert2*/


/*OLDENTRY
double MatSolve(n,M,b)
Solve an nxn system of linear equations Mx=b
using Gaussian elimination with partial pivoting.
leave solution x in b array (destroying original A and b in the process)
Returns determinant.
(code from Paul Heckbert).
*/

double
MatSolve(int4 n,Flotype *A,Flotype *b)
{
   register int4 i,j,k;
   double max,t,det,sum,pivot;	/* keep these double */

#  define swap(a,b,t) {t=a; a=b; b=t;}
#  define a(i,j) A[(i)*n+(j)]

   /*---------- forward elimination ----------*/

   det = 1.;
   for (i=0; i<n; i++) {		/* eliminate in column i */
      max = -1.;
      for (k=i; k<n; k++)		/* find pivot for column i */
         if (fabs(a(k,i))>max) {
            max = fabs(a(k,i));
            j = k;
         }
      if (max<=0.) return(0.);		/* if no nonzero pivot, PUNT */
      if (j!=i) {			/* swap rows i and j */
         for (k=i; k<n; k++)
            swap(a(i,k),a(j,k),t);
         det = -det;
         swap(b[i],b[j],t);		/* swap elements of column vector */
      }
      pivot = a(i,i);
      det *= pivot;
      for (k=i+1; k<n; k++)		/* only do elems to right of pivot */
         a(i,k) /= pivot;

      /* we know that a(i,i) will be set to 1, so why bother to do it? */
      b[i] /= pivot;
      for (j=i+1; j<n; j++) {		/* eliminate in rows below i */
         t = a(j,i);			/* we're gonna zero this guy */
         for (k=i+1; k<n; k++)		/* subtract scaled row i from row j */
            a(j,k) -= a(i,k)*t;		/* (ignore k<=i, we know they're 0) */
         b[j] -= b[i]*t;
      }
   }

   /*---------- back substitution ----------*/

   for (i=n-1; i>=0; i--) {		/* solve for x[i] (put it in b[i]) */
      sum = b[i];
      for (k=i+1; k<n; k++)		/* really a(i,k)*x[k] */
         sum -= a(i,k)*b[k];
      b[i] = sum;
   }

   return(det);

#  undef swap
#  undef a

} /*solve*/


/*OLDENTRY
MatRandom(M,m,n)
Fills M with random numbers between -1..1 
(for testing matrix inversion routines).
*/

void MatRandom(Flotype *M,int4 m,int4 n)
{
    int4 i,l;

    l = n*m;
    for( i=0; i < l; i++ ) *M++ = rndf11();
}


/****************************************************************
 ************************  Vector  ******************************
 ****************************************************************/

#define VecKey 'V'

/*OLDENTRY
Flotype *V = VecAlloc(int4 n)
Allocates storage and a type-checking field for
a vector of length n.
*/

Flotype *
VecAlloc(int4 n)
{
    Flotype *V;

    V = (Flotype *)malloc((n+3) * sizeof(Flotype));

    /* set type-checking keys before,after the data
     * [key,length,...data...,key]
     */
    *((int4 *)(V+0)) = (int4)VecKey;
    *((int4 *)(V+1)) = n;	/* length */
    *((int4 *)(V+2+n)) = (int4)VecKey;

    return(V+2);
} /*Alloc*/


/*OLDENTRY
VecFree(V)
Frees V; complains if V was not allocated with VecAlloc.
*/

void VecFree(Flotype *V)
{
    int4 n;

    V -= 2;
    if (*((int4 *)(V+0)) != VecKey)  Zcodeerror("VecFree");

    n = *((int4 *)(V+1));
    Ztrace(("VecFree recovered length %d\n",n));

    if (*((int4 *)(V+2+n)) != (int4)VecKey)
	Zcodeerror("VecFree:corrupted matrix");

    free((char *)V);
} /*Free*/



/*OLDENTRY
VecPrint(char *msg,V,n)
Prints V, prefaced by msg.
*/

void VecPrint(char *msg,Flotype V[],int4 n)
{
    int4 i;
    printf("%s\n",msg);
    printf("[ ");
    for(i=0;i<n;i++) printf("%.3f  ",V[i]);
    printf(" ]\n");
} /*Print*/



/*OLDENTRY
VecCopy(V1,V2,n)
Copies V1 to V2.
*/

void VecCopy(Flotype *V1,Flotype *V2,int4 n)
{
    Zbcopy((char *)V1,(char *)V2,n*sizeof(Flotype));
} /*Copy*/


/*OLDENTRY
MatVecMul(M,V1, V2,N)
Postmultiplies matrix M by vector V1, result in V2.
*/

void MatVecMul(Flotype *M,Flotype *V1, Flotype *V2,int4 N)
{
    register int4 i,j;
    register Flotype sum;

    for( i=0; i < N; i++ ) {
	sum = 0.0;
	for( j=0; j < N; j++ ) sum += *M++ * V1[j];
	V2[i] = sum;
    }

} /*MatVecMul*/



/*OLDENTRY
VecRandom(V,n)
Fills V with random numbers between -1..1 
(for testing purposes).
*/

void VecRandom(Flotype *V,int4 n)
{
    int4 i;

    for( i=0; i < n; i++ ) *V++ = rndf11();
}


/*@DOCFINIT*/

#ifdef TESTIT /*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*% cc -DTESTIT -DFLTPREC -Dztrace % -lZ -lm -o matrixTST %*/

main()
{
    Flotype *m,*v,*v2;
    m = MatAlloc(5,5);
    MatFree(m);

    v = VecAlloc(2);
    v2 = VecAlloc(2);

    VecRandom(v,2);
    VecCopy(v,v2,2);
    VecPrint("v: ",v,2); VecPrint("copied: ",v2,2);

    VecFree(v);
    VecFree(v2);
}

#endif /*TESTIT %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
