// Pyramid.java - abstract single channel pyramid
// (c) jplewis 1999-2000

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA  02111-1307, USA.
//
// Primary author contact info:  www.idiom.com/~zilla  zilla@computer.org

// modified
// f01		restrictop

import java.util.ArrayList;
import java.io.*;

import GR.*;
import ZS.zmath;
import zlib.*;

/**
 * Abstract single channel pyramid
 * 
 * Store in an arraylist so that new interpolated-up layers can be inserted.
 * _levels.get(0) is highest resolution.
 */
public class Pyramid implements iPyramid
{
  ArrayList	_levels;		// list of gr images
  int		_xres;			// size of the original image,
  int		_yres;			// not necessarily a power of two

  iPyramidOp	_restrictOp = null;	// optional, store the pyramid op here

  final boolean	trace	= true;

  //static String[] exts = new String[]{".r",".g",".b",".a"};

  public Pyramid(String name, gr img)
  {
    setup(name, img, null);
  } //constructor


  public Pyramid(String name, gr img, iPyramidOp op)
  {
    setup(name, img, op);
  } //constructor


  /**
   * build a pyramid with same size as o.
   * @param name
   * @param o
   */
  public Pyramid(String name, final Pyramid o)
  {
    setup(name, o._xres, o._yres, o._restrictOp);
    zliberror.assert(_xres == o._xres);
    zliberror.assert(_yres == o._yres);
  } //constructor


  // setup for a particular image,
  // and initialize the base level
  private void setup(String name, gr img, iPyramidOp op)
  {
    setup(name, img.xres(), img.yres(), op);

    // make base level by padding original out to power of 2
    // pad by pixel replication rather than zero because
    // otherwise algorithms that are deciding something based on the
    // range of image derivatives will be thrown off by a large internal edge
    int pyres = ((gr)_levels.get(0)).yres();
    int pxres = ((gr)_levels.get(0)).xres();
    short[] buf = ((gr)_levels.get(0)).allocrow();

    for( int y=0; y<_yres; y++ ) {
      img.rdblk(0,y,_xres-1,y, buf);

      // pad on rhs
      for( int x=_xres; x < pxres; x++ ) buf[x] = buf[_xres-1];

      ((gr)_levels.get(0)).wrblk(0,y,pxres-1,y, buf);
    }

    // pad on bottom
    for( int y=_yres; y < pyres; y++ ) 
      ((gr)_levels.get(0)).wrblk(0,y,pxres-1,y, buf);

    if (op != null) restrict(op);

  } //setup


  // setup an empty pyramid of a particular size
  private void setup(String name, int xres, int yres, iPyramidOp op)
  {
    _xres = xres;
    _yres = yres;
    int po2res = _xres < _yres ? _xres : _yres;
    int nlevels = zmath.nextpowerof2(po2res);
    int xres2 = 1<<zmath.nextpowerof2(_xres);
    int yres2 = 1<<zmath.nextpowerof2(_yres);
    // force square here if desired

    _levels = new ArrayList();
    for(int i=0; i < nlevels; i++ ) {
      System.out.println(i +": " + xres2 +" x " + yres2);
      _levels.add(new grShort(name +"-"+ i, xres2, yres2));
      xres2 = xres2 / 2;
      yres2 = yres2 / 2;
    }

    _restrictOp = op;
  } //setup

  //----------------------------------------------------------------
  //----------------------------------------------------------------

  public int getNlevels()
  {
    return _levels.size();
  }

  public int getXres(int level)
  {
    gr g = (gr)_levels.get(level);
    return g.xres();
  }

  public int getYres(int level)
  {
    gr g = (gr)_levels.get(level);
    return g.yres();
  }

  //----------------------------------------------------------------

  gr getLevel(int level)
  {
    gr g = (gr)_levels.get(level);
    return g;
  }

  //----------------------------------------------------------------

  /**
   * (re)build the pyramid, using the operator given to the constructor
   */
  public void restrict()
  {
    iPyramidOp p = _restrictOp;
    for( int il=1; il < _levels.size(); il++ ) {
      if (trace) System.out.println("restrict1 " + (il-1) +" -> "+il);
      restrict1(p, (gr)_levels.get(il-1), (gr)_levels.get(il));
    } //il
  } //restrict


  // build the pyramid, assuming _level0 is filled in
  public void restrict(iPyramidOp p)
  {
    for( int il=1; il < _levels.size(); il++ ) {
      if (trace) System.out.println("restrict1 " + (il-1) +" -> "+il);
      restrict1(p, (gr)_levels.get(il-1), (gr)_levels.get(il));
    } //il
  } //restrict


  public void restrict1(iPyramidOp p, gr hi, gr low)
  {
    int lyres = low.yres();
    int lxres = low.xres();
    int hyres = hi.yres();
    int hxres = hi.xres();

    short[] hibuf1 = hi.allocrow();
    short[] hibuf2 = hi.allocrow();
    short[] lowbuf = low.allocrow();

    for( int iy=0; iy < lyres; iy++ ) {
      hi.rdblk(0, 2*iy, hxres-1, 2*iy, hibuf1);
      hi.rdblk(0, 2*iy+1, hxres-1, 2*iy+1, hibuf2);
      for( int ix=0; ix < lxres; ix++ ) {
	int ix2 = 2*ix;
	int ix21 = ix2 + 1;
	lowbuf[ix] = p.call(hibuf1[ix2], hibuf1[ix21],
			    hibuf2[ix2], hibuf1[ix21]);
      }
      low.wrblk(0, iy, lxres-1, iy, lowbuf);
    } //iy
  } //restrict1


  /**
   * save in a ppm file
   */
  public void save(String path)
    throws IOException
  {
    gr base = getLevel(0);

    int xres = base.xres();
    byte buf[] = new byte[Math.max(xres, 64)];

    int yres = 0;
    for( int ilevel=0; ilevel < _levels.size(); ilevel++ ) {
      gr g = getLevel(ilevel);
      yres += g.yres();
    }

    BufferedOutputStream f
      = new BufferedOutputStream(new FileOutputStream(path));
    zlib.writeString(f, new String("P5\n"));
    zlib.writeString(f, new String("# Pyramid pgm\n"));
    String s = xres + " " + yres + "\n";
    zlib.writeString(f, s);
    zlib.writeString(f, new String("255\n"));

    float scale = 255.f / gr.DTMAX;
    int y = 0;
    for( int ilevel=0; ilevel < _levels.size(); ilevel++ ) {
      gr g = getLevel(ilevel);
      int gyres = g.yres();
      int gxres = g.xres();
      short/*GRDTYPE*/[] gbuf = g.allocrow();
      for( int x=0; x < xres; x++ ) { buf[x] = 0; }

      for( int iy=0; iy < gyres; iy++ ) {
	g.rdblk(0, iy, gxres-1, iy, gbuf);
	for( int ix=0; ix < gxres; ix++ ) {
	  int v = (int)(0.5f + scale * gbuf[ix]);
	  // 128 -> -128,
	  // 255 -> -1;
	  if (v > 127) v = v - 256;
	  buf[ix] = (byte)v;
	}
	f.write(buf, 0, xres);

	y++;
      } //iy(level)
	  
    } //level
    zliberror.assert(y == yres);

    //for( int x=0; x < xres; x++ ) { buf[x] = 0; }
    //for( ; y < yres; y++ ) f.write(buf, 0, xres);
    f.close();
    
  } //save

  //----------------------------------------------------------------
  // static methods for a color component array of pyramids
  //----------------------------------------------------------------

  /**
   * static, make an array of pyramid components from this image
   */
  public static Pyramid[] mkpyramid(gr[] image, iPyramidOp op)
  {
    Pyramid[] pyr = new Pyramid[image.length];

    for( int ic=0; ic < image.length; ic++ ) {
      pyr[ic] = new Pyramid(image[ic].name() + "." + ic, image[ic], op);
    }

    return pyr;
  } //mkpyramid


  /**
   * static, return gr[] of all the components of the pyramid at this level
   */
  public static gr[] getLevel(Pyramid[] pyr, int level)
  {
    int ncomp = pyr.length;
    gr[] g = new gr[ncomp];
    for( int ic=0; ic < ncomp; ic++ ) {
      g[ic] = pyr[ic].getLevel(level);
    }
    return g;
  } //getLevel


  /**
   * static
   */
  static public void restrict(Pyramid[] pyr)
  {
    for( int ic=0; ic < pyr.length; ic++ ) {
      pyr[ic].restrict();
    }
  } //restrict


  /**
   * static.  Save to set of pgm files
   * TODO: SAVE TO A RGB PPM FILE INSTEAD
   */
  static public void save(Pyramid[] pyr, String pathprefix)
    throws IOException
  {
    for( int ic=0; ic < pyr.length; ic++ ) {
      pyr[ic].save(pathprefix + ic + ".ppm");
    }
  } //save


} //Pyramid
